#!/usr/local/bin/perl

#┌─────────────────────────────────
#│ CafeLog : index.cgi - 2023/10/09
#│ copyright (c) kentweb, 1997-2023
#│ https://www.kent-web.com/
#└─────────────────────────────────

# モジュール宣言
use strict;
use CGI::Carp qw(fatalsToBrowser);
use vars qw($cgi %in %cf);
use lib "./lib";
use CGI::Simple;
use CGI::Session;
use Digest::SHA::PurePerl qw(sha256_base64);

# 設定ファイル認識
require './init.cgi';
%cf = set_init();

# データ受理
$CGI::Simple::DISABLE_UPLOADS = 0;
$CGI::Simple::POST_MAX = $cf{maxdata};
$cgi = new CGI::Simple;
error($cgi->cgi_error) if ($cgi->cgi_error);
%in = parse_form($cgi);

# 基本設定
set_base();

# 各種処理
if ($in{base_mgr}) { require "./lib/base_mgr.pl"; base_mgr(); }
if ($in{tmpl_mgr}) { require "./lib/tmpl_mgr.pl"; tmpl_mgr(); }
if ($in{side_mgr}) { require "./lib/side_mgr.pl"; side_mgr(); }
if ($in{cate_mgr}) { require "./lib/cate_mgr.pl"; cate_mgr(); }
if ($in{link_mgr}) { require "./lib/link_mgr.pl"; link_mgr(); }
if ($in{tbcm_add}) { require "./lib/tbcm_add.pl"; tbcm_add(); }
if ($in{tbcm_mgr}) { require "./lib/tbcm_mgr.pl"; tbcm_mgr(); }
if ($in{pass_mgr}) { require "./lib/pass_mgr.pl"; pass_mgr(); }
if ($in{img_mgr})  { require "./lib/img_mgr.pl";  img_mgr(); }
if ($in{img_pop})  { require "./lib/img_mgr.pl";  img_pop(); }
if ($in{ref_chg})  { require "./lib/ref_chg.pl";  ref_chg(); }
page_mng();

#-----------------------------------------------------------
#  ページ管理
#-----------------------------------------------------------
sub page_mng {
	require "./lib/page_mgr.pl";
	$in{page_mgr}++;
	page_mgr();
}

#-----------------------------------------------------------
#  基本設定
#-----------------------------------------------------------
sub set_base {
	# 基本情報認証
	open(IN,"$cf{datadir}/base.dat") or error("open err: base.dat");
	while(<IN>) {
		chomp;
		my ($key,$val) = split(/\t/);
		
		$cf{$key} = $val;
	}
	close(IN);
	
	# 認証チェック
	require './lib/auth_axs.pl';
	check_auth();
}

#-----------------------------------------------------------
#  HTMLヘッダー
#-----------------------------------------------------------
sub header {
	my ($ttl,$css) = @_;
	
	print <<EOM;
Content-type: text/html; charset=utf-8

<!doctype html>
<html lang="ja">
<head>
<meta charset="utf-8">
<link href="$cf{htmlurl}/tool/admin.css" rel="stylesheet">
<title>$ttl</title>
EOM

	if ($css eq 'uppy') {
		print qq|<link href="https://transloadit.edgly.net/releases/uppy/v1.2.0/uppy.min.css" rel="stylesheet">\n|;
	}
	
	print <<EOM;
</head>
<body>
<div id="head">
	<div id="subject"><img src="$cf{htmlurl}/tool/star.png" class="icon"> Cafe Log ::</div>
	<div id="logoff"><a href="$cf{index_cgi}?sid=$in{sid}&mode=logoff" onclick="return confirm('ログオフしますか？');">ログオフ</a></div>
</div>
EOM
}

#-----------------------------------------------------------
#  フッター
#-----------------------------------------------------------
sub footer {
	print <<EOM;
<div id="bot"></div>
<div id="foot"></div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  エラー画面
#-----------------------------------------------------------
sub error {
	my $msg = shift;
	
	header("ERROR!");
	menu_btn();
	print <<EOM;
<div id="body">
<div id="main">
<div class="ta-c" style="width:100%">
<div class="err-box ta-c">
	<p style="font-size:120%;">
		<img src="$cf{htmlurl}/tool/error.png" alt="" class="icon"> <b>ERROR!</b>
	</p>
	<div class="err">$msg</div>
	<input type="button" value="前画面に戻る" onclick="history.back()">
</div>
</div>
</div>
</div>
EOM
	footer();
}

#-----------------------------------------------------------
#  完了メッセージ
#-----------------------------------------------------------
sub echo_msg {
	my ($msg,$mode,$fl) = @_;
	
	header("処理完了");
	menu_btn();
	print <<EOM;
<div id="body">
<div id="main">
<div class="ta-c">
<form action="$cf{index_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="$mode" value="1">
EOM

	if ($in{cate} ne '') {
		print qq|<input type="hidden" name="cate" value="$in{cate}">\n|;
	}
	if ($fl ne '') {
		print qq|<input type="hidden" name="file" value="$fl">\n|;
	}
	
	print <<EOM;
<div class="msg-box">
	<p style="margin-bottom:2.5em"><img src="$cf{htmlurl}/tool/tick.png" alt="" class="icon"> <b>$msg</b></p>
	<input type="submit" value="元の画面へ戻る" class="bigbtn">
</div>
</form>
</div>
</div>
</div>
EOM
	footer();
}

#-----------------------------------------------------------
#  メニューボタン
#-----------------------------------------------------------
sub menu_btn {
	my @navi = (
		'base_mgr:基本設定',
		'page_mgr:ページ管理',
		'img_mgr:画像管理',
		'tbcm_add:コメント/TB承認',
		'tbcm_mgr:コメント/TB管理',
		'cate_mgr:カテゴリ管理',
		'side_mgr:サイド/フッタ管理',
		'link_mgr:リンク管理',
		'ref_chg:変更反映',
		'tmpl_mgr:テンプレート編集',
		'pass_mgr:パスワード管理',
	);
	
	print <<EOM;
<div id="menu">
<div id="navi">
<form action="$cf{index_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<table class="menu-btn">
EOM

	for (@navi) {
		my ($key,$val) = split(/:/);
		
		if ($in{$key}) {
			print qq|<tr><th><input type="button" value="$val" disabled></th></tr>\n|;
		} else {
			print qq|<tr><th><input type="submit" name="$key" value="$val"></th></tr>\n|;
		}
	}
	
	print <<EOM;
<tr>
	<th><input type="button" value="サイト表示" onclick="window.open('$cf{htmlurl}/','_blank')"></th>
</tr>
</table>
</form>
</div>
</div>
EOM
}

#-----------------------------------------------------------
#  nicEdit定義
#-----------------------------------------------------------
sub nic_edit {
	print <<'EOM';
<script>
var html_editor = new Array();
var html_editor_loaded = false;
function loadHtmlEditor() {
	if (!html_editor_loaded) {
		var scripts = document.getElementsByTagName('script');
		for (var i=0; i<scripts.length; i++) {
			if (scripts[i].src.search(/nicEdit\.js$/) != -1) return true;
		}
		var header = document.getElementsByTagName('head')[0];
		var loader = document.createElement('script');
		loader.setAttribute('type','text/javascript');
EOM

	# パス指定
	print qq|loader.setAttribute('src','$cf{htmlurl}/tool/nicEdit.js');\n|;
	
	print <<'EOM';
		header.appendChild(loader);
		html_editor_loaded = true;
	}
}
function updateHtmlEditor(this_id) {
	if (html_editor[this_id] != null) {
		html_editor[this_id].updateInstance(this_id);
	}
}
function toggleHtmlEditor(open,this_id,panel_type) {
	if (open) {
		if (typeof nicEditor == "undefined") {
			loadHtmlEditor();
			setTimeout(function() { toggleHtmlEditor(open,this_id,panel_type) },500);
		} else {
			if (html_editor[this_id] == null) html_editor[this_id] = new nicEditor({panelType : panel_type}).panelInstance(this_id);
		}
	} else if (html_editor[this_id] != null) {
		html_editor[this_id].removeInstance(this_id);
		html_editor[this_id] = null;
	}
}
</script>
EOM
}

#-----------------------------------------------------------
#  タグ復元
#-----------------------------------------------------------
sub chg_tag {
	local($_) = @_;
	
	s/&amp;/&/g;
	s/&quot;/"/g;
	s/&lt;/</g;
	s/&gt;/>/g;
	s/&#39;/'/g;
	$_;
}

#-----------------------------------------------------------
#  繰越ボタン作成
#-----------------------------------------------------------
sub make_pgbtn {
	my ($i,$pg,$mode,$cate) = @_;
	
	# 引数
	my $param = "sid=$in{sid}&$mode=1";
	if ($cate) { $param .= "&cate=$in{cate}"; }
	
	# ページ繰越数定義
	$cf{pg_max} ||= 50;
	my $next = $pg + $cf{list_max};
	my $back = $pg - $cf{list_max};
	
	# ページ繰越ボタン作成
	my @pg;
	if ($back >= 0 || $next < $i) {
		my $flg;
		my ($w,$x,$y,$z) = (0,1,0,$i);
		while ($z > 0) {
			if ($pg == $y) {
				$flg++;
				push(@pg,qq!<li><span>$x</span></li>!);
			} else {
				push(@pg,qq!<li><a href="$cf{index_cgi}?pg=$y&$param">$x</a></li>!);
			}
			$x++;
			$y += $cf{list_max};
			$z -= $cf{list_max};
			
			if ($flg) { $w++; }
			last if ($w >= 5 && @pg >= 10);
		}
	}
	while( @pg >= 11 ) { shift(@pg); }
	my $ret = join('', @pg);
	if ($back >= 0) {
		$ret = qq!<li><a href="$cf{index_cgi}?pg=$back&$param">&laquo;</a></li>\n! . $ret;
	}
	if ($next < $i) {
		$ret .= qq!<li><a href="$cf{index_cgi}?pg=$next&$param">&raquo;</a></li>\n!;
	}
	return qq|<ul class="pageNav01">\n| . $ret . qq|</ul>\n|;
}

#-----------------------------------------------------------
#  フォームデコード
#-----------------------------------------------------------
sub parse_form {
	my $cgi = shift;
	
	my %in;
	foreach ( $cgi->param() ) {
		my $val = $cgi->param($_);
		
		# エスケープ
		if ($_ ne 'upfile') {
			$val =~ s/&/&amp;/g;
			$val =~ s/"/&quot;/g;
			$val =~ s/</&lt;/g;
			$val =~ s/>/&gt;/g;
			$val =~ s/'/&#39;/g;
		}
		$in{$_} = $val;
	}
	return %in;
}

#-----------------------------------------------------------
#  件名文字カット for UTF-8
#-----------------------------------------------------------
sub cut_str {
	my ($str,$all) = @_;
	$str =~ s/<.*?>//g;
	
	# UTF-8定義
	my $byte1 = '[\x00-\x7f]';
	my $byte2 = '[\xC0-\xDF][\x80-\xBF]';
	my $byte3 = '[\xE0-\xEF][\x80-\xBF]{2}';
	my $byte4 = '[\xF0-\xF7][\x80-\xBF]{3}';
	
	my $i = 0;
	my $ret;
	while ($str =~ /($byte1|$byte2|$byte3|$byte4)/gx) {
		$i++;
		$ret .= $1;
		last if ($i >= $all);
	}
	return $ret;
}

#-----------------------------------------------------------
#  Digest::SHA256 暗号
#-----------------------------------------------------------
sub encrypt {
	my $plain = shift;
	
	my @str = ('a' .. 'z', 'A' .. 'Z', 0 .. 9);
	my $salt; 
	for (1 .. 8) { $salt .= $str[int(rand(@str))]; }
	
	# 変換
	return $salt . sha256_base64($salt . $plain);
}

#-----------------------------------------------------------
#  Digest::SHA256 照合
#-----------------------------------------------------------
sub decrypt {
	my ($crypt,$plain) = @_;
	
	# 照合
	my $salt = substr($crypt,0,8);
	return $crypt eq ($salt . sha256_base64($salt . $plain)) ? 1 : 0;
}

