#!/usr/local/bin/perl

#┌────────────────────────────
#│ HAPPY CART : admin.cgi - 2023/07/08
#│ copyright (c) KentWeb, 1997-2023
#│ https://www.kent-web.com/
#└────────────────────────────

# モジュール宣言
use strict;
use CGI::Carp qw(fatalsToBrowser);
use vars qw(%in %cf);
use lib "./lib";
use CGI::Minimal;
use CGI::Session;
use DBM::Deep;
use Digest::SHA::PurePerl qw(sha256_base64);

# 設定ファイル認識
require "./init.cgi";
%cf = set_init();

# データ受理
CGI::Minimal::max_read_size($cf{maxdata});
my $cgi = CGI::Minimal->new;
error('容量オーバー') if ($cgi->truncated);
%in = parse_form($cgi);

# 認証
require "./lib/login.pl";
auth_login();

# 処理分岐
if ($in{data_new}) { data_new(); }
if ($in{data_mgr}) { data_mgr(); }
if ($in{base_mgr}) { base_mgr(); }
if ($in{list_log}) { list_log(); }
if ($in{edit_law}) { edit_law(); }
if ($in{pass_mgr}) { pass_mgr(); }
menu_top();

#-----------------------------------------------------------
#  管理トップ
#-----------------------------------------------------------
sub menu_top {
	header("管理トップ");
	print <<EOM;
<div id="body">
<div class="ta-c">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<table id="menu-tbl">
<tr>
	<th></th>
	<th class="val">処理メニュー</th>
</tr><tr>
	<td class="ta-c"><input type="submit" name="data_new" value="選択"></td>
	<td class="val">商品の新規登録</td>
</tr><tr>
	<td class="ta-c"><input type="submit" name="data_mgr" value="選択"></td>
	<td class="val">商品の修正/削除</td>
</tr><tr>
	<td class="ta-c"><input type="submit" name="base_mgr" value="選択"></td>
	<td class="val">基本設定</td>
</tr><tr>
	<td class="ta-c"><input type="submit" name="list_log" value="選択"></td>
	<td class="val">注文ログの閲覧</td>
</tr><tr>
	<td class="ta-c"><input type="submit" name="edit_law" value="選択"></td>
	<td class="val">特定商取引法の編集</td>
</tr><tr>
	<td class="ta-c"><input type="submit" name="pass_mgr" value="選択"></td>
	<td class="val">パスワード管理</td>
</tr><tr>
	<td class="ta-c"><input type="submit" name="logoff" value="選択"></td>
	<td class="val">ログオフ</td>
</tr>
</table>
</form>
</div>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  商品登録フォーム
#-----------------------------------------------------------
sub data_new {
	if ($in{job} eq "new") { new_data(); }
	
	my $data = shift;
	my ($item,$price,$size,$col,$memo,$code,$cate,$rtax,$ex1,$w1,$h1,$ex2,$w2,$h2,$ex3,$w3,$h3,$ex4,$w4,$h4,$ex5,$w5,$h5) = split(/\t/,$data);
	$memo =~ s/<br>/\n/g;
	my %pic = (1 => $ex1, 2 => $ex2, 3 => $ex3, 4 => $ex4, 5 => $ex5);
	
	# 引数
	my $mode = $in{data_new} ? 'data_new' : 'data_mgr';
	my $job  = $in{job} eq "" ? 'new' : 'edit2';
	
	# 在庫認識
	my %zan = read_stock() if ($cf{stock} == 1);
	
	# 詳細
	my $detail;
	if ($in{no}) {
		open(IN,"$cf{datadir}/detail/$in{no}.dat");
		$detail = join('',<IN>);
		close(IN);
		
		$detail =~ s/\n//g;
		$detail = encode_tag($detail);
	}
	
	header("登録フォーム","sn");
	back_btn();
	print <<EOM;
<p class="ttl">■ 商品登録フォーム</p>
<div id="body">
<ul>
<li class="note"><span>&nbsp;</span>のついた項目は入力必須です。
<li>画像は [主] が表示され、[副] は詳細画面で表記可能になります。
</ul>
<form action="$cf{admin_cgi}" method="post" enctype="multipart/form-data">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="$mode" value="1">
<input type="hidden" name="job" value="$job">
<input type="hidden" name="no" value="$in{no}">
<table class="form-tbl">
<tr>
	<th class="item"><span>商品コード</span></t>
	<td>
		<input type="text" name="code" size="30" style="ime-mode:inactive" value="$code">
		（英数字又はハイフンで指定）
	</td>
</tr><tr>
	<th class="item"><span>商品名</span></th>
	<td><input type="text" name="item" size="40" style="ime-mode:active" value="$item"></td>
</tr><tr>
	<th class="item"><span>カテゴリ</span></th>
	<td>
		<select name="cate">
		<option value="">▼選択
EOM

	for ( @{$cf{category}} ) {
		my ($key,$val) = split(/:/);
		
		if ($cate == $key) {
			print qq|<option value="$key" selected>$val\n|;
		} else {
			print qq|<option value="$key">$val\n|;
		}
	}
	
	print <<EOM;
		</select>
	</td>
</tr><tr>
	<th class="item"><span>金額</span></th>
	<td>
		<input type="text" name="price" size="20" style="ime-mode:inactive" value="$price"> 円<br>
EOM

	if ($cf{tax_per} > 0) {
		if ($rtax == 1) {
			print qq|<input type="checkbox" name="redtax" value="1" checked>\n|;
		} else {
			print qq|<input type="checkbox" name="redtax" value="1">\n|;
		}
		print qq|軽減税率対象\n|;
	}
	
	print <<EOM;
	</td>
</tr>
EOM

	# 在庫
	if ($cf{stock} == 1) {
		if (!defined $zan{$in{no}}) { $zan{$in{no}} = ""; }
		
		print qq|<tr><th class="item">在庫</th>|;
		print qq|<td><input type="text" name="stk" size="6" style="ime-mode:inactive" value="$zan{$in{no}}"></td></tr>\n|;
		print qq|<input type="hidden" name="now" value="$zan{$in{no}}">\n|;
	}
	
	print <<EOM;
<tr>
	<th class="item">$cf{op1}</th>
	<td>
		（スペースで区切り、複数指定。例: S M L）<br>
		<input type="text" name="size" size="40" value="$size">
	</td>
</tr><tr>
  <th class="item">$cf{op2}</th>
  <td>
	（スペースで区切り、複数指定。例: 赤 青 黄）<br>
	<input type="text" name="color" size="40" value="$col"></td>
</tr><tr>
  <th class="item">画像</th>
  <td nowrap>
	（GIF/JPEG/PNG）<br>
EOM

	my @head = qw(主 主 副 副 副 副);
	for (1 .. 5) {
		print qq|[$head[$_]] <input type="file" name="upfile$_" size="35">\n|;
		if ($pic{$_}) {
			print qq|&nbsp;[<a href="$cf{imgurl}/$in{no}-$_$pic{$_}" target="_blank">画像</a>]\n|;
			print qq|<input type="checkbox" name="imgdel$_" value="1">削除\n|;
		}
		print "<br>\n";
	}
	
	print <<EOM;
	</td>
</tr><tr>
	<th class="item">商品説明</th>
	<td><textarea name="memo" cols="60" rows="3">$memo</textarea></td>
</tr><tr>
	<th class="item">詳細内容</th>
	<td>
		（詳細画面のみに表示されます）<br>
		<textarea name="detail" id="detail">$detail</textarea>
	</td>
</tr>
</table>
<input type="submit" value="送信する">
</form>
</div>
<script>
\$('#detail').summernote({
	height: 240,
	width: 800,
	lang: "ja-JP",
	fontNames: ["YuGothic","Yu Gothic","Hiragino Kaku Gothic Pro","Meiryo","sans-serif", "Arial","Arial Black","Comic Sans MS","Courier New","Helvetica Neue","Helvetica","Impact","Lucida Grande","Tahoma","Times New Roman","Verdana"],
});
</script>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  商品の登録修正
#-----------------------------------------------------------
sub data_mgr {
	my $db = DBM::Deep->new(
		file    => "$cf{datadir}/db/data.db",
	);
	
	# 修正フォーム
	if ($in{job} eq "edit" && $in{no}) {
		my $data = $db->get($in{no});
		data_new($data);
	
	# 修正実行
	} elsif ($in{job} eq "edit2") {
		edit_data();
	
	# 削除実行
	} elsif ($in{job} eq "dele" && $in{no}) {
		dele_data();
	
	# ソート
	} elsif ($in{job} eq "sort") {
		sort_data();
	}
	
	header("登録内容メンテ");
	back_btn();
	print <<EOM;
<p class="ttl">■ 登録内容メンテ</p>
<div id="body">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="data_mgr" value="1">
<ul>
<li>切替ボタンでカテゴリ毎に絞り込み表示できます。
<li>ソートの場合は、カテゴリを【全データ】としてから「順番」の数字を変更します。
</ul>
カテゴリ：
<select name="cate">
<option value="">【全データ】
EOM

	my %cat;
	for ( @{$cf{category}} ) {
		my ($key,$val) = split(/:/);
		$cat{$key} = $val;
		
		if ($in{cate} eq $key) {
			print qq|<option value="$key" selected>$val\n|;
		} else {
			print qq|<option value="$key">$val\n|;
		}
	}
	
	print <<EOM;
</select>
<input type="submit" value="切替">
</form>
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="data_mgr" value="1">
<input type="hidden" name="cate" value="$in{cate}">
処理：
<select name="job">
<option value="edit">修正
<option value="sort">ソート
<option value="dele">削除</select>
<input type="submit" value="送信する">
<table class="form-tbl">
<tr>
	<th nowrap>選択</th>
	<th nowrap>順番</th>
	<th nowrap>コード</th>
	<th nowrap>カテゴリ</th>
	<th nowrap>商品名</th>
	<th nowrap>単価</th>
EOM

	# 在庫機能のとき
	my %zan;
	if ($cf{stock}) {
		%zan = read_stock();
		print "<th nowrap>在庫</th>";
	}
	
	print "</tr>\n";
	
	# 商品情報読み出し
	my $i = 0;
	open(IN,"$cf{datadir}/index.dat");
	while(<IN>) {
		my ($no,undef) = split(/,/);
		my $data = $db->get($no);
		my ($item,$price,$size,$col,$memo,$code,$cate,$rtax,$ex1,$w1,$h1,$ex2,$w2,$h2,$ex3,$w3,$h3,$ex4,$w4,$h4,$ex5,$w5,$h5) = split(/\t/,$data);
		next if ($in{cate} ne '' && $in{cate} != $cate);
		
		$price = comma($price);
		$i++;
		print qq|<td class="ta-c"><input type="checkbox" name="no" value="$no"></td>|;
		print qq|<td><input type="text" name="sort:$no" value="$i" size="3" style="ime-mode:inactive"></td>|;
		print qq|<td>$code</td><td>$cat{$cate}</td><td>$item</td>|;
		print qq|<td class="ta-r">$price円</td>|;
		
		if ($cf{stock}) {
			if (!defined $zan{$no}) { $zan{$no} = 0; }
			print qq|<td class="ta-r">$zan{$no}</td>|;
		}
		
		print "</tr>\n";
	}
	close(IN);
	
	print <<EOM;
</table>
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  新規登録
#-----------------------------------------------------------
sub new_data {
	# 入力チェック
	check_input();
	
	my $db = DBM::Deep->new(
		file    => "$cf{datadir}/db/data.db",
	);
	
	# キー作成
	my $key = make_key();
	if ($in{code} eq "") { $in{code} = $key; }
	if ($in{cate} eq "") { $in{cate} = 0; }
	
	$in{memo} =~ s/\t+$//g;
	$in{memo} =~ s/\t/<br>/g;
	
	# 画像アップ
	my ($ex1,$w1,$h1,$ex2,$w2,$h2,$ex3,$w3,$h3,$ex4,$w4,$h4,$ex5,$w5,$h5) = upload($key)
			if ($in{upfile1} or $in{upfile2} or $in{upfile3} or $in{upfile4} or $in{upfile5});
	
	# DB追加
	$db->put($key => "$in{item}\t$in{price}\t$in{size}\t$in{color}\t$in{memo}\t$in{code}\t$in{cate}\t$in{redtax}\t$ex1\t$w1\t$h1\t$ex2\t$w2\t$h2\t$ex3\t$w3\t$h3\t$ex4\t$w4\t$h4\t$ex5\t$w5\t$h5");
	
	# 詳細
	$in{detail} = decode_tag($in{detail});
	$in{detail} =~ s|</p>|</p>\n|ig;
	open(DAT,"> $cf{datadir}/detail/$key.dat");
	print DAT $in{detail};
	close(DAT);
	
	# index
	open(OUT,">> $cf{datadir}/index.dat");
	print OUT "$key,$in{cate}\n";
	close(OUT);
	
	# 在庫更新
	if ($cf{stock}) {
		my ($flg,@file);
		open(DAT,"+< $cf{datadir}/stock.dat") or error("open err: stock.dat");
		eval "flock(DAT,2);";
		while (<DAT>) {
			my ($no,undef) = split(/,/);
			if ($no == $key) {
				$flg++;
				$_ = "$no,$in{stk}\n";
			}
			push(@file,$_);
		}
		if (!$flg) { push(@file,"$key,$in{stk}\n"); }
		seek(DAT,0,0);
		print DAT @file;
		truncate(DAT,tell(DAT));
		close(DAT);
	}
	# 完了
	message("商品登録を完了しました","data_new");
}

#-----------------------------------------------------------
#  削除処理
#-----------------------------------------------------------
sub dele_data {
	my $db = DBM::Deep->new(
		file    => "$cf{datadir}/db/data.db",
	);
	
	# 削除情報
	my %del;
	for ( $cgi->param('no') ) {
		$del{$_}++;
		
		my $data = $db->get($_);
		my ($item,$price,$size,$col,$memo,$code,$cate,$rtax,$ex1,$w1,$h1,$ex2,$w2,$h2,$ex3,$w3,$h3,$ex4,$w4,$h4,$ex5,$w5,$h5) = split(/\t/,$data);
		my $i;
		for my $ex ($ex1,$ex2,$ex3,$ex4,$ex5) {
			$i++;
			if ($ex) { unlink("$cf{imgdir}/$_-$i$ex"); }
		}
		$db->delete($_);
	}
	
	# index
	my @log;
	open(DAT,"+< $cf{datadir}/index.dat");
	while(<DAT>) {
		my ($no,undef) = split(/,/);
		next if (defined $del{$no});
		
		push(@log,$_);
	}
	seek(DAT,0,0);
	print DAT @log;
	truncate(DAT,tell(DAT));
	close(DAT);
	
	# 在庫
	if ($cf{stock}) {
		my @data;
		open(DAT,"+< $cf{datadir}/stock.dat") or error("open err: stock.dat");
		eval "flock(DAT,2);";
		while (<DAT>) {
			my ($no,undef) = split(/,/);
			next if (defined $del{$no});
			
			push(@data,$_);
		}
		seek(DAT,0,0);
		print DAT @data;
		truncate(DAT,tell(DAT));
		close(DAT);
	}
}

#-----------------------------------------------------------
#  修正処理
#-----------------------------------------------------------
sub edit_data {
	# 入力チェック
	check_input();
	
	my $db = DBM::Deep->new(
		file    => "$cf{datadir}/db/data.db",
	);
	
	# DB抽出
	my $data = $db->get($in{no});
	my ($item,$price,$size,$col,$memo,$code,$cate,$rtax,$ex1,$w1,$h1,$ex2,$w2,$h2,$ex3,$w3,$h3,$ex4,$w4,$h4,$ex5,$w5,$h5) = split(/\t/,$data);
	my %im = (
		1 => [$ex1,$w1,$h1],
		2 => [$ex2,$w2,$h2],
		3 => [$ex3,$w3,$h3],
		4 => [$ex4,$w4,$h4],
		5 => [$ex5,$w5,$h5],
	);
	
	# 画像アップ
	my ($ex1n,$w1n,$h1n,$ex2n,$w2n,$h2n,$ex3n,$w3n,$h3n,$ex4n,$w4n,$h4n,$ex5n,$w5n,$h5n) = upload($in{no})
			if ($in{upfile1} or $in{upfile2} or $in{upfile3} or $in{upfile4} or $in{upfile5});
	my %up = (
		1 => [$ex1n,$w1n,$h1n],
		2 => [$ex2n,$w2n,$h2n],
		3 => [$ex3n,$w3n,$h3n],
		4 => [$ex4n,$w4n,$h4n],
		5 => [$ex5n,$w5n,$h5n],
	);
	
	for my $i (1 .. 5) {
		if ($up{$i}[0] ne '') {
			if ($up{$i}[0] ne $im{$i}[0]) {
				unlink("$cf{imgdir}/$in{no}-$i$im{$i}[0]");
			}
			for (0..2) {
				$im{$i}[$_] = $up{$i}[$_];
			}
		
		} elsif ($in{"imgdel$i"} == 1) {
			unlink("$cf{imgdir}/$in{no}-$i$im{$i}[0]");
			for (0..2) {
				$im{$i}[$_] = '';
			}
		}
	}
	
	# 更新
	$db->put($in{no} => "$in{item}\t$in{price}\t$in{size}\t$in{color}\t$in{memo}\t$in{code}\t$in{cate}\t$in{redtax}\t$im{1}[0]\t$im{1}[1]\t$im{1}[2]\t$im{2}[0]\t$im{2}[1]\t$im{2}[2]\t$im{3}[0]\t$im{3}[1]\t$im{3}[2]\t$im{4}[0]\t$im{4}[1]\t$im{4}[2]\t$im{5}[0]\t$im{5}[1]\t$im{5}[2]");
	
	# 詳細
	$in{detail} = decode_tag($in{detail});
	$in{detail} =~ s|</p>|</p>\n|ig;
	open(DAT,"> $cf{datadir}/detail/$in{no}.dat");
	print DAT $in{detail};
	close(DAT);
	
	# index
	if ($in{cate} ne $cate) {
		my @log;
		open(DAT,"+< $cf{datadir}/index.dat");
		while(<DAT>) {
			my ($no,$cat) = split(/,/);
			
			if ($in{no} == $no) {
				$_ = "$no,$in{cate}\n";
			}
			push(@log,$_);
		}
		seek(DAT,0,0);
		print DAT @log;
		truncate(DAT,tell(DAT));
		close(DAT);
	}
	
	# 在庫更新
	if ($cf{stock} && $in{now} != $in{stk}) {
		my ($flg,@file);
		open(DAT,"+< $cf{datadir}/stock.dat") or error("open err: stock.dat");
		eval "flock(DAT,2);";
		while (<DAT>) {
			my ($no,undef) = split(/,/);
			
			if ($no == $in{no}) {
				$flg++;
				$_ = "$no,$in{stk}\n";
			}
			push(@file,$_);
		}
		if (!$flg) { push(@file,"$in{no},$in{stk}\n"); }
		seek(DAT,0,0);
		print DAT @file;
		truncate(DAT,tell(DAT));
		close(DAT);
	}
	
	# 完了メッセージ
	message("商品を修正しました","data_mgr");
}

#-----------------------------------------------------------
#  並替処理
#-----------------------------------------------------------
sub sort_data {
	my (@log,@tmp);
	open(DAT,"+< $cf{datadir}/index.dat");
	while (<DAT>) {
		my ($no,undef) = split(/,/);
		
		push(@log,$_);
		push(@tmp,$in{"sort:$no"});
	}
	
	# ソート
	@log = @log[sort {$tmp[$a] <=> $tmp[$b]} 0 .. $#tmp];
	
	# 更新
	seek(DAT,0,0);
	print DAT @log;
	truncate(DAT,tell(DAT));
	close(DAT);
}

#-----------------------------------------------------------
#  基本設定
#-----------------------------------------------------------
sub base_mgr {
	# 更新
	if ($in{submit}) {
		
		$in{news} =~ s/\t+$//;
		$in{news} =~ s/\t/<br>/g;
		
		my (%pay,%cat);
		for ( keys %in ) {
			if (/^pay:(\d+)/) {
				$pay{$1}++;
			} elsif (/^cat:(\d+)/) {
				$cat{$1}++;
			}
		}
		
		# カテゴリ
		$in{categ} = '';
		my $end = 0;
		for ( sort{ $a <=> $b } keys %cat ) {
			next if ($in{"cdel:$_"} == 1);
			$in{categ} .= qq|$_:$in{"cat:$_"}<>|;
			if ($end < $_) { $end = $_; }
		}
		$in{categ} =~ s/(<>)$//;
		if ($in{cat_new}) {
			$end++;
			$in{categ} .= "<>$end:$in{cat_new}";
		}
		
		# 支払方式
		$in{pay} = '';
		for ( sort{ $a <=> $b } keys %pay ) {
			next if ($in{"pdel:$_"} == 1);
			$in{pay} .= qq|$in{"pay:$_"},$in{"cost:$_"}<>|;
		}
		$in{pay} =~ s/(<>)$//;
		if ($in{pay_new}) {
			if ($in{cost_new} eq '') { $in{cost_new} = 0; }
			$in{pay} .= "<>$in{pay_new},$in{cost_new}";
		}
		
		my @pref = qw(北海道 青森県 岩手県 宮城県 秋田県 山形県 福島県 茨城県 栃木県 群馬県 埼玉県 千葉県 東京都 神奈川県 新潟県 富山県 石川県 福井県 山梨県 長野県 岐阜県 静岡県 愛知県 三重県 滋賀県 京都府 大阪府 兵庫県 奈良県 和歌山県 鳥取県 島根県 岡山県 広島県 山口県 徳島県 香川県 愛媛県 高知県 福岡県 佐賀県 長崎県 熊本県 大分県 宮崎県 鹿児島県 沖縄県);
		# 送料
		$in{prefect} = '';
		for (0 .. 46) {
			$in{prefect} .= qq|$pref[$_],$in{"pref:$_"}<>|;
		}
		$in{prefect} =~ s/(<>)$//;
		
		# 更新
		my @log;
		for (qw(title news categ tax_per red_per thumbnail stock pg_max cari_serv upl_max_w upl_max_h img_max_w img_max_h op1 op2 pay delitime prefect)) {
			$in{$_} =~ s/\t//g;
			push(@log,"$_\t$in{$_}\n");
		}
		open(DAT,"> $cf{datadir}/base.dat") or error("write err: base.dat");
		print DAT @log;
		close(DAT);
		
		# 完了メッセージ
		message("基本設定を更新しました。","base_mgr");
	}
	
	# 改行
	$cf{news} =~ s/<br>/\n/g;
	
	# 画面表示
	header("基本設定");
	back_btn();
	print <<EOM;
<p class="ttl">■ 基本設定</p>
<div id="body">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="base_mgr" value="1">
<table class="form-tbl">
<tr>
	<th>サイト名称</th>
	<td><input type="text" name="title" value="$cf{title}" size="50"></td>
</tr><tr>
	<th>お知らせ内容</th>
	<td><textarea name="news" cols="50" rows="3">$cf{news}</textarea></td>
</tr><tr>
	<th>消費税</th>
	<td>
		税方式： <input type="number" name="tax_per" value="$cf{tax_per}" class="w-4"> ％
		（0=内税、0以上で外税）<br>
		軽減税率： <input type="number" name="red_per" value="$cf{red_per}" class="w-4"> ％
		（税方式が 0 以上の場合に有効）
	</td>
</tr><tr>
	<th>商品属性</th>
	<td>
		第1オプション： <input type="text" name="op1" value="$cf{op1}" size="10"><br>
		第2オプション： <input type="text" name="op2" value="$cf{op2}" size="10">
	</td>
</tr><tr>
	<th>画像リサイズ</th>
	<td>
		横：<input type="number" name="upl_max_w" value="$cf{upl_max_w}" class="w-4"> pix &nbsp;
		縦：<input type="number" name="upl_max_h" value="$cf{upl_max_h}" class="w-4"> pix
		（詳細画面）<br>
		横：<input type="number" name="img_max_w" value="$cf{img_max_w}" class="w-4"> pix &nbsp;
		縦：<input type="number" name="img_max_h" value="$cf{img_max_h}" class="w-4"> pix
		（一覧画面）<br>
		自動サムネイル化：
EOM

	my %ox = (1 => 'あり', 0 => 'なし');
	for (1,0) {
		if ($cf{thumbnail} == $_) {
			print qq|<input type="radio" name="thumbnail" value="$_" checked>$ox{$_}\n|;
		} else {
			print qq|<input type="radio" name="thumbnail" value="$_">$ox{$_}\n|;
		}
	}
	
	print <<EOM;
	（ImgResize.pm 使用）
	</td>
</tr><tr>
	<th>在庫機能</th>
	<td>
EOM

	for (1,0) {
		if ($cf{stock} == $_) {
			print qq|<input type="radio" name="stock" value="$_" checked>$ox{$_}\n|;
		} else {
			print qq|<input type="radio" name="stock" value="$_">$ox{$_}\n|;
		}
	}
	
	print <<EOM;
	</td>
</tr><tr>
	<th>表示件数</th>
	<td><input type="number" name="pg_max" value="$cf{pg_max}" class="w-4"> 件
		（ページ当たりの商品表示件数）
	</td>
</tr><tr>
	<th>カテゴリー</th>
	<td>
		<input type="text" name="cat_new" size="20">
		[追加] ← 新規追加する場合<br>
EOM

	for ( split(/<>/,$cf{categ}) ) {
		my ($key,$val) = split(/:/);
		
		print qq|<input type="text" name="cat:$key" value="$val" size="20">\n|;
		print qq|<input type="checkbox" name="cdel:$key" value="1">削除<br>\n|;
	}
	
	print <<EOM;
	</td>
</tr><tr>
	<th>支払方式</th>
	<td>
		（名称＋コスト金額）<br>
		<input type="text" name="pay_new" size="20">
		<input type="number" name="cost_new" class="w-4">
		[追加] ← 新規追加する場合<br>
EOM

	my $i = 0;
	for ( split(/<>/,$cf{pay}) ) {
		my ($pay,$cost) = split(/,/);
		
		print qq|<input type="text" name="pay:$i" value="$pay" size="20">\n|;
		print qq|<input type="number" name="cost:$i" value="$cost" class="w-4">\n|;
		print qq|<input type="checkbox" name="pdel:$i" value="1">削除<br>\n|;
		$i++;
	}
	
	print <<EOM;
	</td>
</tr><tr>
	<th>配達時間</th>
	<td>
		（コンマで区切る）<br>
		<input type="text" name="delitime" value="$cf{delitime}" size="55">
	</td>
</tr><tr>
	<th>送料サービス</th>
	<td><input type="number" name="cari_serv" value="$cf{cari_serv}" class="w-6"> 円
		（買物金額がこれ以上の場合は無料）
	</td>
</tr><tr>
	<th>送料</th>
	<td>
EOM

	my @pref = split(/<>/,$cf{prefect});
	my $i;
	for (0 .. 46) {
		my ($pref,$cost) = split(/,/,$pref[$_]);
		print qq|$pref <input type="number" name="pref:$_" value="$cost" class="w-4"> 円 &nbsp;\n|;
		$i++;
		if ($i % 3 == 0) { print "<br>\n"; }
	}
	
	print <<EOM;
	</td>
</tr>
</table>
<input type="submit" name="submit" value="更新する">
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  注文情報ログ
#-----------------------------------------------------------
sub list_log {
	# ページ
	my $pg = $in{pg} || 0;
	
	# 閲覧
	my $log;
	foreach ( $cgi->param() ) {
		if (/^log:(\d+)/) {
			$log = $1;
			last;
		} elsif (/^pg:(\d+)/) {
			$pg = $1;
			last;
		}
	}
	if ($log) { look_log($log); }
	
	# ログ
	my @dir;
	opendir(DIR,"$cf{datadir}/log");
	while(defined( $_ = readdir(DIR) )) {
		next if (!/(\d{6})\.cgi$/);
		push(@dir,$1);
	}
	closedir(DIR);
	
	$in{date} =~ s/\D//g;
	if (!$in{date}) { $in{date} = $dir[0]; }
	
	# 画面表示
	header("注文情報ログ");
	back_btn();
	print <<EOM;
<p class="ttl">■ 注文情報ログ</p>
<div id="body">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="list_log" value="1">
年月：
<select name="date">
EOM

	foreach ( sort{ $b <=> $a } @dir ) {
		my $val = $_ =~ /^(\d{4})(\d{2})/ ? "$1年$2月" : $_;
		
		if ($in{date} == $_) {
			print qq|<option value="$_" selected>$val\n|;
		} else {
			print qq|<option value="$_">$val\n|;
		}
	}
	
	print <<EOM;
</select>
<input type="submit" value="切替">
<table class="form-tbl">
<tr>
	<th>選択</th>
	<th>日付</th>
	<th>注文番号</th>
</tr>
EOM

	# 表示件数
	my $pg_max = 50;
	
	my $i;
	open(IN,"$cf{datadir}/log/$in{date}.cgi");
	while(<IN>) {
		$i++;
		next if ($i < $pg + 1);
		next if ($i > $pg + $pg_max);
		
		my ($date,$num,undef) = split(/<>/);
		
		print qq|<tr><td><input type="submit" name="log:$num" value="閲覧"></td>|;
		print qq|<td>$date</td>|;
		print qq|<td class="ta-c">$num</td></tr>\n|;
	}
	close(IN);
	
	print "</table>\n";
	
	my $next = $pg + $pg_max;
	my $back = $pg - $pg_max;
	if ($back >= 0) {
		print qq|<input type="submit" name="pg:$back" value="前ページ">\n|;
	}
	if ($next < $i) {
		print qq|<input type="submit" name="pg:$next" value="次ページ">\n|;
	}
	
	print <<EOM;
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  個別ログ
#-----------------------------------------------------------
sub look_log {
	my $no = shift;
	$in{date} =~ s/\D//g;
	
	my $data;
	open(IN,"$cf{datadir}/log/$in{date}.cgi");
	while(<IN>) {
		my ($date,$num,$log) = split(/<>/);
		if ($no == $num) {
			$data = $log;
			last;
		}
	}
	close(IN);
	
	$data =~ s/\t/\n/g;
	
	header("個別ログ");
	print <<EOM;
<div id="body">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="list_log" value="1">
<input type="hidden" name="date" value="$in{date}">
<input type="submit" value="&lt; 戻る">
</form>
<p class="ttl">■ 注文情報ログ &gt; 個別ログ</p>
<pre>
$data
</pre>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  特定商取引法メンテ
#-----------------------------------------------------------
sub edit_law {
	# 新規追加
	if ($in{add}) {
		if (!$in{item} || !$in{text}) {
			error("項目または内容が未記入です");
		}
		
		my ($new,@data);
		open(DAT,"+< $cf{datadir}/law.dat") or error("open err: law.dat");
		while (<DAT>) {
			my ($no,$key,$val) = split(/\t/);
			
			if ($new < $no) { $new = $no; }
			push(@data,$_);
		}
		$new++;
		
		# 追加
		seek(DAT,0,0);
		print DAT @data;
		print DAT "$new\t$in{item}\t$in{text}\n";
		truncate(DAT,tell(DAT));
		close(DATA);
	}
	
	# 削除
	my $del;
	foreach ( $cgi->param() ) {
		if (/^del:(\d+)/) {
			$del = $1;
			last;
		}
	}
	if ($del) {
		my @data;
		open(DAT,"+< $cf{datadir}/law.dat") or error("open err: law.dat");
		while (<DAT>) {
			my ($no,$key,$val) = split(/\t/);
			next if ($del == $no);
			
			push(@data,$_);
		}
		seek(DAT,0,0);
		print DAT @data;
		truncate(DAT,tell(DAT));
		close(DAT);
	
	# 修正
	} elsif ($in{mente}) {
		
		my @data;
		open(DAT,"+< $cf{datadir}/law.dat") or error("open err: law.dat");
		while (<DAT>) {
			my ($no,$key,$val) = split(/\t/);
			next if ($in{del} && $in{del} == $no);
			
			push(@data,"$no\t$in{\"item$no\"}\t$in{\"text$no\"}\n");
		}
		seek(DAT,0,0);
		print DAT @data;
		truncate(DAT,tell(DAT));
		close(DAT);
	}
	
	# 画面表示
	header("特商法編集");
	back_btn();
	print <<EOM;
<p class="ttl">■ 特商法編集</p>
<div id="body">
<ul>
<li>新規に項目を追加します。（末尾に追加）
</ul>
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="edit_law" value="1">
<table class="form-tbl">
<tr>
	<th>項目</th>
	<td><input type="text" name="item" size="35"></td>
</tr><tr>
	<th>内容</th>
	<td><textarea name="text" cols="45" rows="2"></textarea></td>
</tr>
</table>
<input type="submit" name="add" value="追加する">
<ul>
<li>各項目を修正＆削除します。（改行無効）
</ul>
<table class="form-tbl">
<tr>
	<th>削除</th>
	<th>項目</th>
	<th>内容</th>
</tr>
EOM

	open(IN,"$cf{datadir}/law.dat") or error("open err: law.dat");
	while (<IN>) {
		chomp;
		my ($no,$key,$val) = split(/\t/);
		
		print qq|<tr><td valign="top" class="ta-c"><input type="submit" name="del:$no" value="削除"></td>|;
		print qq|<td valign="top"><input type=text name="item$no" value="$key" size="25"></td>|;
		print qq|<td><textarea name="text$no" cols="45" rows="2">$val</textarea></td></tr>\n|;
	}
	close(IN);
	
	print <<EOM;
</table>
<input type="submit" name="mente" value="修正する">
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  アップロード
#-----------------------------------------------------------
sub upload {
	my $num = shift;
	
	# サムネイル機能
	require './lib/resize.pl' if ($cf{thumbnail});
	
	my @ret;
	for my $i (1 .. 5) {
		
		# ファイル名
		my $fname = $cgi->param_filename("upfile$i");
		if ($fname =~ /(\.jpe?g|\.png|\.gif)$/i) {
			my $ex = $1;
			if ($ex eq '.jpeg') { $ex = '.jpg'; }
			$ex =~ tr/A-Z/a-z/;
			
			# アップファイル定義
			my $imgfile;
			if ($cf{thumbnail}) {
				$imgfile = "$cf{imgdir}/$$-$i$ex";
			} else {
				$imgfile = "$cf{imgdir}/$num-$i$ex";
			}
			
			# 書き込み
			open(OUT,"> $imgfile") or error("画像アップ失敗");
			binmode(OUT);
			print OUT $in{"upfile$i"};
			close(OUT);
			
			# 画像サイズ取得
			my ($w,$h);
			if ($ex eq ".jpg") { ($w,$h) = j_size($imgfile); }
			elsif ($ex eq ".gif") { ($w,$h) = g_size($imgfile); }
			elsif ($ex eq ".png") { ($w,$h) = p_size($imgfile); }
			
			# サムネイル作成
			if ($cf{thumbnail}) {
				($w,$h) = resize($w,$h,$cf{upl_max_w},$cf{upl_max_h});
				$ex = '.jpg';
				my $thumb = "$cf{imgdir}/$num-$i$ex";
				img_resize($imgfile,$thumb,$w,$h);
				unlink($imgfile);
			}
			push(@ret,($ex,$w,$h));
		} else {
			push(@ret,('','',''));
		}
	}
	return @ret;
}

#-----------------------------------------------------------
#  JPEGサイズ認識
#-----------------------------------------------------------
sub j_size {
	my $jpg = shift;
	
	my ($h,$w,$t);
	open(IMG,"$jpg");
	binmode(IMG);
	read(IMG,$t,2);
	while (1) {
		read(IMG,$t,4);
		my ($m,$c,$l) = unpack("a a n", $t);
		
		if ($m ne "\xFF") {
			$w = $h = 0;
			last;
		} elsif ((ord($c) >= 0xC0) && (ord($c) <= 0xC3)) {
			read(IMG,$t,5);
			($h, $w) = unpack("xnn", $t);
			last;
		} else {
			read(IMG,$t,($l - 2));
		}
	}
	close(IMG);
	
	return ($w,$h);
}

#-----------------------------------------------------------
#  GIFサイズ認識
#-----------------------------------------------------------
sub g_size {
	my $gif = shift;
	
	my $data;
	open(IMG,"$gif");
	binmode(IMG);
	sysread(IMG,$data,10);
	close(IMG);
	
	if ($data =~ /^GIF/) { $data = substr($data, -4); }
	my $w = unpack("v", substr($data,0,2));
	my $h = unpack("v", substr($data,2,2));
	
	return ($w,$h);
}

#-----------------------------------------------------------
#  PNGサイズ認識
#-----------------------------------------------------------
sub p_size {
	my $png = shift;
	
	my $data;
	open(IMG,"$png");
	binmode(IMG);
	read(IMG,$data,24);
	close(IMG);
	
	my $w = unpack("N", substr($data,16,20));
	my $h = unpack("N", substr($data,20,24));
	
	return ($w,$h);
}

#-----------------------------------------------------------
#  メニュー戻りボタン
#-----------------------------------------------------------
sub back_btn {
	print <<EOM;
<div id="body">
<div class="back-btn">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="submit" value="&lt; メニュー">
</form>
</div>
EOM
}

#-----------------------------------------------------------
#  エラー
#-----------------------------------------------------------
sub error {
	my $err = shift;
	
	header("ERROR!");
	print <<EOM;
<div id="body">
<div id="err">
<h3>ERROR!</h3>
<p class="err">$err</p>
<p><input type="button" value="前画面に戻る" onclick="history.back()"></p>
</div>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  完了メッセージ
#-----------------------------------------------------------
sub message {
	my ($msg,$mode) = @_;
	
	header("処理完了");
	print <<EOM;
<div id="body">
<div class="msg-body ta-c">
<hr width="350">
<p class="msg">$msg</p>
<hr width="350">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="submit" value="管理TOPに戻る">
@{[ $mode ? qq|<input type="submit" name="$mode" value="初期画面に戻る">| : "" ]}
</form>
</div>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  HTMLヘッダー
#-----------------------------------------------------------
sub header {
	my ($ttl,$sn) = @_;
	
	print <<EOM;
Content-type: text/html; charset=utf-8

<!doctype html>
<html lang="ja">
<head>
<meta charset="UTF-8">
<link href="$cf{cmnurl}/admin.css" rel="stylesheet">
<title>$ttl</title>
EOM

	if ($sn eq 'sn') {
		print qq|<script src="$cf{cmnurl}/jquery/jquery.min.js"></script>\n|;
		print qq|<link href="$cf{cmnurl}/summernote/summernote-lite.min.css" rel="stylesheet">\n|;
		print qq|<script src="$cf{cmnurl}/summernote/summernote-lite.min.js"></script>\n|;
		print qq|<script src="$cf{cmnurl}/summernote/lang/summernote-ja-JP.js"></script>\n|;
	}
	
	print <<EOM;
</head>
<body>
<div id="head">:: HAPPY CART 管理画面 ::</div>
EOM
}

#-----------------------------------------------------------
#  入力チェック
#-----------------------------------------------------------
sub check_input {
	my $err;
	if ($in{item} eq "") { $err .= '商品名の入力モレです<br>'; }
	if ($in{price} eq "") { $err .= '単価の入力モレです<br>'; }
	if ($in{price} =~ /[^\d\,]/) {
		$err .= '単価には半角数字とコンマ以外は入力できません<br>';
	}
	$in{price} =~ s/,//g;
	if ($in{code} =~ /[^A-Za-z0-9\-]/) {
		$err .= '商品コードは英数字とハイフンのみです<br>';
	}
	if ($in{cate} eq "") {
		$err .= 'カテゴリ選択は必須です<br>';
	}
	if ($cf{stock} && $in{stk} eq "") {
		$err .= "在庫数が未入力です<br>";
	}
	error($err) if ($err);
	
	$in{memo} =~ s/&lt;/</g;
	$in{memo} =~ s/&gt;/>/g;
	$in{memo} =~ s/&quot;/"/g;
	$in{memo} =~ s/&amp;/&/g;
}

#-----------------------------------------------------------
#  タグ復元
#-----------------------------------------------------------
sub decode_tag {
	local($_) = @_;
	
	s/&gt;/>/g;
	s/&lt;/</g;
	s/&quot;/"/g;
	s/&amp;/&/g;
	s/&#39;/'/g;
	
	return $_;
}

#-----------------------------------------------------------
#  タグ変換
#-----------------------------------------------------------
sub encode_tag {
	local($_) = @_;
	
	s/&/&amp;/g;
	s/>/&gt;/g;
	s/</&lt;/g;
	s/"/&quot;/g;
	s/'/&#39;/g;
	
	return $_;
}

#-----------------------------------------------------------
#  キー番号作成
#-----------------------------------------------------------
sub make_key {
	open(DAT,"+< $cf{datadir}/key.dat") or error("open err: key.dat");
	eval "flock(DAT,2);";
	my $num = <DAT> + 1;
	seek(DAT,0,0);
	print DAT $num;
	truncate(DAT,tell(DAT));
	close(DAT);
	
	return $num;
}

#-----------------------------------------------------------
#  フォームデコード
#-----------------------------------------------------------
sub parse_form {
	my $cgi = shift;
	
	my %in;
	for ( $cgi->param() ) {
		my $val = $cgi->param($_);
		
		if (!/^upfile[1-5]$/) {
			
			# 無効化
			$val =~ s/&/&amp;/g;
			$val =~ s/</&lt;/g;
			$val =~ s/>/&gt;/g;
			$val =~ s/"/&quot;/g;
			$val =~ s/'/&#39;/g;
			$val =~ s/\r\n/\t/g;
			$val =~ s/\n/\t/g;
		}
		$in{$_} = $val;
	}
	return %in;
}
