#!/usr/local/bin/perl

#┌─────────────────────────────────
#│ CLIP BOARD : clip.cgi - 2022/05/28
#│ copyright (c) kentweb, 1997-2022
#│ https://www.kent-web.com/
#└─────────────────────────────────

# モジュール宣言
use strict;
use CGI::Carp qw(fatalsToBrowser);
use lib "./lib";
use CGI::Minimal;

# 設定ファイル認識
require "./init.cgi";
my %cf = set_init();

# データ受理
CGI::Minimal::max_read_size($cf{maxdata});
my $cgi = CGI::Minimal->new;
error('容量オーバー') if ($cgi->truncated);
my %in = parse_form($cgi);

# 処理分岐
if ($in{mode} eq 'find') { find_data(); }
if ($in{mode} eq 'note') { note_page(); }
if ($in{mode} eq 'view') { view_page(); }
bbs_list();

#-----------------------------------------------------------
#  記事リスト画面
#-----------------------------------------------------------
sub bbs_list {
	del_form() if ($in{del});
	
	# ページ数定義
	my $pg = $in{pg} || 0;
	
	# 表示データ認識
	my ($i,@log);
	open(IN,"$cf{datadir}/log.cgi") or error("open err: log.cgi");
	while (<IN>) {
		$i++;
		next if ($i < $pg + 1);
		next if ($i > $pg + $cf{pg_max});
		
		push(@log,$_);
	}
	close(IN);
	
	# ページ繰越ボタン作成
	my $pgbtn = make_btn($i,$pg);
	
	# クッキー取得
	my @cook = get_cookie();
	$cook[2] ||= 'http://';
	
	# テンプレート読み込み
	open(IN,"$cf{tmpldir}/bbs.html") or error("open err: bbs.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	# 文字置換
	$tmpl =~ s|!icon:(\w+\.\w+)!|<img src="$cf{cmnurl}/$1" alt="$1" class="icon">|g;
	$tmpl =~ s/!([a-z]+_cgi|homepage|cmnurl|bbs_title)!/$cf{$1}/g;
	$tmpl =~ s/!page_btn!/$pgbtn/g;
	$tmpl =~ s/!reg-nam!/$cook[0]/g;
	$tmpl =~ s/!reg-eml!/$cook[1]/g;
	$tmpl =~ s/!reg-url!/$cook[2]/g;
	
	# 画像認証作成
	my ($str_plain,$str_crypt);
	if ($cf{use_captcha} > 0) {
		require $cf{captcha_pl};
		($str_plain,$str_crypt) = cap::make($cf{captcha_key},$cf{cap_len});
		$tmpl =~ s/!str_crypt!/$str_crypt/g;
	} else {
		$tmpl =~ s|<!-- captcha -->.+?<!-- /captcha -->||s;
	}
	
	# テンプレート分割
	my ($head,$loop,$foot) = $tmpl =~ m|(.+)<!-- loop -->(.+?)<!-- /loop -->(.+)|s
			? ($1,$2,$3)
			: error('テンプレート不正');
	
	# 画面展開
	print "Content-type: text/html; charset=utf-8\n\n";
	print $head;
	
	# 表示データ展開
	foreach my $log (@log) {
		my ($no,$date,$name,$eml,$sub,$com,$url,$host,$pw,$ex,$w,$h,$tim,$chk) = split(/<>/,$log);
		$name = qq|<a href="mailto:$eml">$name</a>| if ($eml);
		$url &&= qq|<a href="$url" target="_blank"><img src="$cf{cmnurl}/home.png" class="icon" alt="$no$ex"></a>|;
		$com = auto_link($com) if ($cf{autolink});
		
		# 添付あり
		if ($ex) {
			# 確認待ち
			if ($cf{imgcheck} && $chk != 1) {
				my $size = get_size("$cf{imgdir}/$no$ex");
				$com .= qq|<p>[添付]: 認証待ち ($size)</p>|;
			
			# 添付画像
			} elsif ($ex =~ /jpg$|gif$|png$/) {
				my $op;
				if (-f "$cf{imgdir}/$no-s$ex") {
					$op = qq|src="$cf{imgurl}/$no-s$ex"|;
				} else {
					($w,$h) = resize($w,$h);
					$op = qq|src="$cf{imgurl}/$no$ex" width="$w" height="$h"|;
				}
				$com .= qq|<p><a href="$cf{imgurl}/$no$ex" target="_blank"><img $op border="0"></a></p>|;
			
			# 画像以外
			} else {
				my $size = get_size("$cf{imgdir}/$no$ex");
				$com .= qq|<p>添付: <a href="$cf{imgurl}/$no$ex" target="_blank"><img src="$cf{cmnurl}/clip.png" class="icon"></a> ($size)</p>|;
			}
		}
		# 記事書き出し
		my $tmp = $loop;
		$tmp =~ s/!num!/$no/g;
		$tmp =~ s/!sub!/$sub/g;
		$tmp =~ s/!name!/$name/g;
		$tmp =~ s/!email!/$eml/g;
		$tmp =~ s/!home!/$url/g;
		$tmp =~ s/!date!/$date/g;
		$tmp =~ s/!comment!/$com/g;
		$tmp =~ s/!pg!/$pg/g;
		print $tmp;
	}
	
	# フッタ
	footer($foot);
}

#-----------------------------------------------------------
#  ワード検索
#-----------------------------------------------------------
sub find_data {
	# 引数取得
	$in{word} =~ s|<br>||g;
	$in{cond} =~ s/\D//g;
	$in{pg} =~ s/\D//g;
	$in{pg} ||= 0;
	
	# 検索条件プルダウン
	my %op = (1 => 'AND', 0 => 'OR');
	my $op_cond;
	foreach (1,0) {
		if ($in{cond} eq $_) {
			$op_cond .= qq|<option value="$_" selected>$op{$_}</option>\n|;
		} else {
			$op_cond .= qq|<option value="$_">$op{$_}</option>\n|;
		}
	}
	
	# 検索実行
	my ($i,@log) = search($in{word},$in{cond}) if ($in{word} ne '');
	
	# 検索用
	my $ewd = $in{word};
	$ewd =~ s/(\W)/"%".unpack("H2",$1)/ego;
	$ewd =~ tr/ /+/;
	my $param = "&amp;mode=find&amp;word=$ewd&amp;cond=$in{cond}";
	
	# ページ繰越ボタン作成
	my $pgbtn = make_btn($i,$in{pg},$param);
	
	# テンプレート読み込み
	open(IN,"$cf{tmpldir}/find.html") or error("open err: find.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	# 文字置換
	$tmpl =~ s/!(bbs_cgi|bbs_title|cmnurl)!/$cf{$1}/g;
	$tmpl =~ s/<!-- op_cond -->/$op_cond/;
	$tmpl =~ s/!word!/$in{word}/;
	$tmpl =~ s/!page_btn!/$pgbtn/;
	
	# テンプレート分割
	my ($head,$loop,$foot) = $tmpl =~ m|(.+)<!-- loop -->(.+?)<!-- /loop -->(.+)|s
			? ($1,$2,$3)
			: error("テンプレート不正");
	
	# 画面展開
	print "Content-type: text/html; charset=utf-8\n\n";
	print $head;
	
	foreach (@log) {
		my ($no,$date,$name,$eml,$sub,$com,$url,$host,$pw,$ex,$w,$h,$tim,$chk) = split(/<>/);
		$name = qq|<a href="mailto:$eml">$name</a>| if ($eml);
		$url &&= qq|<a href="$url" target="_blank"><img src="$cf{cmnurl}/home.png" class="icon" alt="home"></a>|;
		$com = auto_link($com) if ($cf{autolink});
		
		# 添付あり
		my $clip;
		if ($ex) {
			my $size = get_size("$cf{imgdir}/$no$ex");
			if ($cf{imgcheck} && $chk != 1) {
				$clip = qq|<div class="clip">[添付]: 認証待ち ($size)</div>|;
			} else {
				$clip = qq|<div class="clip">[添付]: <a href="$cf{imgurl}/$no$ex" target="_blank">$no$ex</a> ($size)</div>|;
			}
		}
		# 記事書き出し
		my $tmp = $loop;
		$tmp =~ s/!num!/$no/g;
		$tmp =~ s/!sub!/$sub/g;
		$tmp =~ s/!date!/$date/g;
		$tmp =~ s/!name!/$name/g;
		$tmp =~ s/!url!/$url/g;
		$tmp =~ s/!comment!/$com/g;
		$tmp =~ s/<!-- clip -->/$clip/;
		print $tmp;
	}
	
	# フッタ
	footer($foot);
}

#-----------------------------------------------------------
#  検索実行
#-----------------------------------------------------------
sub search {
	my ($word,$cond) = @_;
	
	# キーワードを配列化
	$word =~ s/\x81\x40/ /g;
	my @wd = split(/\s+/,$word);
	
	# UTF-8定義
	my $byte1 = '[\x00-\x7f]';
	my $byte2 = '[\xC0-\xDF][\x80-\xBF]';
	my $byte3 = '[\xE0-\xEF][\x80-\xBF]{2}';
	my $byte4 = '[\xF0-\xF7][\x80-\xBF]{3}';
	
	# 検索処理
	my ($i,@log);
	open(IN,"$cf{datadir}/log.cgi") or error("open err: log.cgi");
	while (<IN>) {
		my ($no,$date,$nam,$eml,$sub,$com,$url,$hos,$pw,$ex,$w,$h,$tim,$chk) = split(/<>/);
		
		my $flg;
		foreach my $wd (@wd) {
			if ("$nam $eml $sub $com $url" =~ /^(?:$byte1|$byte2|$byte3|$byte4)*?\Q$wd\E/i) {
				$flg++;
				if ($cond == 0) { last; }
			} else {
				if ($cond == 1) { $flg = 0; last; }
			}
		}
		next if (!$flg);
		
		$i++;
		next if ($i < $in{pg} + 1);
		next if ($i > $in{pg} + $cf{pg_max});
		
		push(@log,$_);
	}
	close(IN);
	
	# 検索結果
	return ($i,@log);
}

#-----------------------------------------------------------
#  留意事項
#-----------------------------------------------------------
sub note_page {
	# 許可拡張子（表示用）
	my $ftype = ext_file();
	
	open(IN,"$cf{tmpldir}/note.html") or error("open err: note.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	$tmpl =~ s/!filetype!/$ftype/g;
	$tmpl =~ s/!(bbs_title|cmnurl)!/$cf{$1}/g;
	
	print "Content-type: text/html; charset=utf-8\n\n";
	print $tmpl;
	exit;
}

#-----------------------------------------------------------
#  アルバムページ
#-----------------------------------------------------------
sub view_page {
	my $pg = $in{pg} eq '' ? '0' : $in{pg};
	
	my ($i,@list);
	open(IN,"$cf{datadir}/log.cgi") or error("open err: log.cgi");
	while (<IN>) {
		my ($no,$date,$name,$eml,$sub,$com,$url,$host,$pw,$ex,$w,$h,$tim,$chk) = split(/<>/);
		next if ($cf{imgcheck} && $chk != 1);
		next if (!$w);
		
		$i++;
		next if ($i < $pg + 1);
		next if ($i > $pg + $cf{pg_max});
		
		push(@list,"$no\t$sub\t$ex\t$w\t$h");
	}
	close(IN);
	
	my $pager = make_btn($i,$pg,"&amp;mode=view");
	
	open(IN,"$cf{tmpldir}/view.html") or error("open err: view.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	$tmpl =~ s/!(bbs_title|bbs_cgi|cmnurl)!/$cf{$1}/g;
	$tmpl =~ s|!pager!|$pager|g;
	
	my ($head,$loop,$foot) = $tmpl =~ m|(.+)<!-- loop -->(.+?)<!-- /loop -->(.+)|s
			? ($1,$2,$3)
			: error('テンプレート不正');
	
	print "Content-type: text/html; charset=utf-8\n\n";
	print $head;
	
	for (@list) {
		my ($no,$sub,$ex,$w,$h) = split(/\t/);
		($w,$h) = resize($w,$h);
		
		my $tmp = $loop;
		$tmp =~ s|!image!|<img src="$cf{imgurl}/$no$ex" alt="$sub" title="$sub" width="$w" height="$h">|;
		print $tmp;
	}
	
	print $foot;
	exit;
}

#-----------------------------------------------------------
#  フッター
#-----------------------------------------------------------
sub footer {
	my $foot = shift;
	
	# 著作権表記（削除・改変禁止）
	my $copy = <<EOM;
<p style="margin-top:2em;text-align:center;font-family:Verdana,Helvetica,Arial;font-size:10px;">
	- <a href="https://www.kent-web.com/" target="_top">ClipBoard</a> -
</p>
EOM

	if ($foot =~ /(.+)(<\/body[^>]*>.*)/si) {
		print "$1$copy$2\n";
	} else {
		print "$foot$copy\n";
		print "</body></html>\n";
	}
	exit;
}

#-----------------------------------------------------------
#  自動リンク
#-----------------------------------------------------------
sub auto_link {
	my $text = shift;
	
	$text =~ s/(s?https?:\/\/([\w-.!~*'();\/?:\@=+\$,%#]|&amp;)+)/<a href="$1" target="_blank">$1<\/a>/g;
	return $text;
}

#-----------------------------------------------------------
#  クッキー取得
#-----------------------------------------------------------
sub get_cookie {
	# クッキー取得
	$ENV{HTTP_COOKIE} =~ /$cf{cookie_id}=(.+);?/;
	my $cookie = $1;
	
	# URLデコード
	my @cook;
	foreach ( split(/<>/,$cookie) ) {
		s/%([0-9A-Fa-f][0-9A-Fa-f])/pack("H2",$1)/eg;
		s/[&"'<>]//g;
		
		push(@cook,$_);
	}
	return @cook;
}

#-----------------------------------------------------------
#  取り扱いファイル
#-----------------------------------------------------------
sub ext_file {
	my $ext;
	foreach ( keys %cf ) {
		next if ($_ !~ /^ok_(\w+)/);
		my $file = $1;
		next if (!$cf{$_});
		
		$file =~ tr/a-z/A-Z/;
		$ext .= "$file, ";
	}
	$ext =~ s/, $//;
	
	return $ext;
}

#-----------------------------------------------------------
#  繰越ボタン作成
#-----------------------------------------------------------
sub make_btn {
	my ($i,$pg,$param) = @_;
	
	# ページ繰越数定義
	$cf{pg_max} ||= 10;
	my $next = $pg + $cf{pg_max};
	my $back = $pg - $cf{pg_max};
	
	# ページ繰越ボタン作成
	my @pg;
	if ($back >= 0 || $next < $i) {
		my $flg;
		my ($w,$x,$y,$z) = (0,1,0,$i);
		while ($z > 0) {
			if ($pg == $y) {
				$flg++;
				push(@pg,qq!<span class="page active">$x</span>\n!);
			} else {
				push(@pg,qq!<a href="$cf{bbs_cgi}?pg=$y$param" class="page gradient">$x</a>\n!);
			}
			$x++;
			$y += $cf{pg_max};
			$z -= $cf{pg_max};
			
			if ($flg) { $w++; }
			last if ($w >= 5 && @pg >= 10);
		}
	}
	while( @pg >= 11 ) { shift(@pg); }
	my $ret = join('',@pg);
	if ($back >= 0) {
		$ret = qq!<a href="$cf{bbs_cgi}?pg=$back$param" class="page gradient">&laquo;</a>\n! . $ret;
	}
	if ($next < $i) {
		$ret .= qq!<a href="$cf{bbs_cgi}?pg=$next$param" class="page gradient">&raquo;</a>\n!;
	}
	
	# 結果を返す
	return $ret ? qq|<div class="pagination">\n$ret</div>| : '';
}

#-----------------------------------------------------------
#  ファイル容量
#-----------------------------------------------------------
sub get_size {
	my $file = shift;
	
	my $size = -s $file || 0;
	return $size > 1024 ? comma(int($size/1024)) . 'KB' : $size . 'Bytes';
}

#-----------------------------------------------------------
#  桁区切り
#-----------------------------------------------------------
sub comma {
	local($_) = @_;
	
	1 while s/(.*\d)(\d\d\d)/$1,$2/;
	$_;
}

#-----------------------------------------------------------
#  記事削除確認
#-----------------------------------------------------------
sub del_form {
	my %log;
	open(IN,"$cf{datadir}/log.cgi") or error("open err: log.cgi");
	while (<IN>) {
		my ($no,$date,$nam,$eml,$sub,$com,$url,$hos,$pw,$ex,$w,$h,$tim,$chk) = split(/<>/);
		if ($in{del} == $no) {
			$log{flg}++;
			$log{num}  = $no;
			$log{sub}  = $sub;
			$log{date} = $date;
			$log{name} = $nam;
			last;
		}
	}
	close(IN);
	
	if (!$log{flg}) { error("該当記事が見つかりません"); }
	
	open(IN,"$cf{tmpldir}/dele.html") or error("open err: dele.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	my $pg = $in{pg} || 0;
	$tmpl =~ s/!(\w+_\w+|cmnurl)!/$cf{$1}/g;
	$tmpl =~ s/!(sub|name|date|num)!/$log{$1}/g;
	$tmpl =~ s/!pg!/$pg/g;
	$tmpl =~ s|!icon:(\w+\.\w+)!|<img src="$cf{cmnurl}/$1" alt="$1" class="icon">|g;
	
	print "Content-type: text/html; charset=utf-8\n\n";
	print $tmpl;
	exit;
}

