#!/usr/local/bin/perl

#┌─────────────────────────────────
#│ CLIP BOARD : regist.cgi - 2022/05/28
#│ copyright (c) kentweb, 1997-2022
#│ https://www.kent-web.com/
#└─────────────────────────────────

# モジュール宣言
use strict;
use CGI::Carp qw(fatalsToBrowser);
use lib "./lib";
use CGI::Minimal;

# 設定ファイル認識
require "./init.cgi";
my %cf = set_init();

# データ受理
CGI::Minimal::max_read_size($cf{maxdata});
my $cgi = CGI::Minimal->new;
error('容量オーバー') if ($cgi->truncated);
my %in = parse_form($cgi);

# 処理分岐
if ($in{mode} eq 'regist') { regist(); }
if ($in{mode} eq 'usrdel') { usrdel(); }
error("不明な処理です");

#-----------------------------------------------------------
#  投稿受付処理
#-----------------------------------------------------------
sub regist {
	# チェック
	if ($cf{no_wd}) { no_wd(); }
	if ($cf{jp_wd}) { jp_wd(); }
	if ($cf{urlnum} > 0) { urlnum(); }
	
	# 不要改行カット
	$in{sub}  =~ s|<br>||g;
	$in{name} =~ s|<br>||g;
	$in{pwd}  =~ s|<br>||g;
	$in{captcha} =~ s|<br>||g;
	$in{comment} =~ s|(<br>)+$||g;
	
	# 入力値調整
	if ($in{url} eq 'http://') { $in{url} = ''; }
	$in{sub} ||= "無題";
	
	# フォーム内容をチェック
	my $err;
	if ($in{name} eq "") { $err .= "名前が入力されていません<br>"; }
	if ($in{comment} eq "") { $err .= "コメントが入力されていません<br>"; }
	if ($in{email} ne '' && $in{email} !~ /^[\w\.\-]+\@[\w\.\-]+\.[a-zA-Z]{2,6}$/) {
		$err .= "Ｅメールの入力内容が不正です<br>";
	}
	if ($in{url} ne '' && $in{url} !~ /^https?:\/\/[\w-.!~*'();\/?:\@&=+\$,%#]+$/) {
		$err .= "参照先URLの入力内容が不正です<br>";
	}
	if ($err) { error($err); }
	
	# 投稿キーチェック
	if ($cf{use_captcha} > 0) {
		require $cf{captcha_pl};
		if ($in{captcha} !~ /^\d{$cf{cap_len}}$/) {
			error("投稿キーが入力不備です。<br>投稿フォームに戻って再読込み後、再入力してください");
		}
		
		# 投稿キーチェック
		# -1 : キー不一致
		#  0 : 制限時間オーバー
		#  1 : キー一致
		my $chk = cap::check($in{captcha},$in{str_crypt},$cf{captcha_key},$cf{cap_time},$cf{cap_len});
		if ($chk == 0) {
			error("投稿キーが制限時間を超過しました。<br>投稿フォームに戻って再読込み後、指定の数字を再入力してください");
		} elsif ($chk == -1) {
			error("投稿キーが不正です。<br>投稿フォームに戻って再読込み後、再入力してください");
		}
	}
	
	# 時間取得
	$ENV{TZ} = "JST-9";
	my $now = time;
	my ($min,$hour,$mday,$mon,$year,$wday) = (localtime($now))[1..6];
	my @week = ('Sun','Mon','Tue','Wed','Thu','Fri','Sat');
	my $date = sprintf("%04d/%02d/%02d(%s) %02d:%02d",
				$year+1900,$mon+1,$mday,$week[$wday],$hour,$min);
	
	# ホスト取得
	my ($host,$addr) = get_host();
	
	open(DAT,"+< $cf{datadir}/log.cgi") or error("open err: log.cgi");
	eval "flock(DAT,2);";
	my $log = <DAT>;
	
	my ($no,$nam,$com,$time) = (split(/<>/, $log))[0,2,5,12];
	if ($in{name} eq $nam && $in{comment} eq $com) {
		close(DAT);
		error("二重投稿は禁止です");
	}
	if ($cf{w_regist} && $now - $time < $cf{w_regist}) {
		close(DAT);
		error("連続投稿はもうしばらく時間を置いてからお願い致します");
	}
	
	# 記事No採番
	$no++;
	
	# ファイル添付処理
	my ($ext,$w,$h) = upload($no) if ($in{upfile});
	
	# 削除キー暗号化
	my $crypt = encrypt($in{pwd}) if ($in{pwd} ne "");
	
	# 最大記事数処理
	my $i = 0;
	my @log;
	seek(DAT, 0, 0);
	while (<DAT>) {
		$i++;
		my ($no,$dat,$nam,$eml,$sub,$com,$url,$hos,$pw,$ex) = split(/<>/);
		
		if ($i <= $cf{maxlog} - 1) {
			push(@log,$_);
		} else {
			if ($ex) {
				unlink("$cf{imgdir}/$no$ex");
				unlink("$cf{imgdir}/$no-s$ex") if (-f "$cf{imgdir}/$no-s$ex");
			}
		}
	}
	
	# 新記事
	unshift(@log,"$no<>$date<>$in{name}<>$in{email}<>$in{sub}<>$in{comment}<>$in{url}<>$host<>$crypt<>$ext<>$w<>$h<>$now<>0<>\n");
	
	# 更新
	seek(DAT,0,0);
	print DAT @log;
	truncate(DAT,tell(DAT));
	close(DAT);
	
	# クッキー格納
	set_cookie($in{name},$in{email},$in{url}) if ($in{cookie} == 1);
	
	# メール通知
	my $clip = "$no$ext" if ($ext);
	mail_to($date,$host,$clip) if ($cf{mailing} == 1);
	
	# 完了画面
	message("ありがとうございます。記事を受理しました。");
}

#-----------------------------------------------------------
#  添付アップロード
#-----------------------------------------------------------
sub upload {
	my $no = shift;
	
	# サムネイル機能
	require './lib/thumb.pl' if ($cf{thumbnail});
	
	# 拡張子取得
	my $ext = $cgi->param_filename('upfile') =~ /(\.[^\\\/:\.]+)$/ ? $1 : error("拡張子不明");
	$ext =~ tr/A-Z/a-z/;
	if ($ext eq '.jpeg') { $ext = '.jpg'; }
	if ($ext eq '.text') { $ext = '.txt'; }
	
	# ファイルチェック
	check_upl($ext);
	
	# 添付ファイル定義
	my $upfile = "$cf{imgdir}/$no$ext";
	
	# アップロード書き込み
	open(UP,"+> $upfile") or error("up err: $upfile");
	binmode(UP);
	print UP $in{upfile};
	close(UP);
	
	chmod(0666,$upfile);
	
	# 画像サイズ取得
	my ($flg,$w,$h);
	if ($ext eq ".jpg") { ($w,$h) = j_size($upfile); $flg++; }
	elsif ($ext eq ".gif") { ($w,$h) = g_size($upfile); $flg++; }
	elsif ($ext eq ".png") { ($w,$h) = p_size($upfile); $flg++; }
	
	# サムネイル化
	if ($flg && $cf{thumbnail}) {
		($w,$h) = resize($w,$h);
		my $thumb = "$cf{imgdir}/$no-s$ext";
		make_thumb($upfile,$thumb,$w,$h);
	}
	
	# 結果を返す
	return ($ext,$w,$h);
}

#-----------------------------------------------------------
#  ユーザ記事削除
#-----------------------------------------------------------
sub usrdel {
	# 入力チェック
	$in{num} =~ s/\D//g;
	if ($in{num} eq '' or $in{pwd} eq '') {
		error("削除Noまたは削除キーが入力モレです");
	}
	
	my ($flg,$crypt,$ext,@log);
	open(DAT,"+< $cf{datadir}/log.cgi") or error("open err: log.cgi");
	eval "flock(DAT,2);";
	while (<DAT>) {
		my ($no,$dat,$nam,$eml,$sub,$com,$url,$hos,$pw,$ex,$chk) = split(/<>/);
		
		if ($in{num} == $no) {
			$flg++;
			$crypt = $pw;
			$ext = $ex;
			next;
		}
		push(@log,$_);
	}
	
	if (!$flg or $crypt eq '') {
		close(DAT);
		error("削除キーが設定されていないか、該当記事が見当たりません");
	}
	
	# 削除キーを照合
	if (decrypt($in{pwd},$crypt) != 1) {
		close(DAT);
		error("認証できません");
	}
	
	# ログを更新
	seek(DAT,0,0);
	print DAT @log;
	truncate(DAT,tell(DAT));
	close(DAT);
	
	# 添付削除
	if ($ext) {
		unlink("$cf{imgdir}/$in{num}$ext");
		unlink("$cf{imgdir}/$in{num}-s$ext") if (-f "$cf{imgdir}/$in{num}-s$ext");
	}
	
	message("記事を削除しました");
}

#-----------------------------------------------------------
#  メール送信
#-----------------------------------------------------------
sub mail_to {
	my ($date,$host,$clip) = @_;
	
	# メールタイトル定義
	require './lib/jacode.pl';
	my $msub = mime_unstructured_header("BBS: $in{sub}");
	
	# コメント内の改行復元
	my $com = $in{comment};
	$com =~ s|<br>|\n|g;
	$com =~ s/&lt;/>/g;
	$com =~ s/&gt;/</g;
	$com =~ s/&quot;/"/g;
	$com =~ s/&amp;/&/g;
	$com =~ s/&#39;/'/g;
	
	# URL情報
	if ($in{url} eq 'http://') { $in{url} = ''; }
	
	# メール本文を定義
	my $mbody = <<EOM;
掲示板に投稿がありました。

投稿日：$date
ホスト：$host

件名  ：$in{sub}
お名前：$in{name}
E-mail：$in{email}
URL   ：$in{url}
添付  ：$clip

$com
EOM

	my $body;
	for my $tmp ( split(/\n/,$mbody) ) {
		jcode::convert(\$tmp,'jis','utf8');
		$body .= "$tmp\n";
	}
	
	# メールアドレスがない場合は管理者メールに置き換え
	$in{email} ||= $cf{mailto};
	
	# sendmailコマンド
	my $scmd = "$cf{sendmail} -t -i";
	if ($cf{sendm_f}) { $scmd .= " -f $in{email}"; }
	
	# 送信
	open(MAIL,"| $scmd") or error("送信失敗");
	print MAIL "To: $cf{mailto}\n";
	print MAIL "From: $in{email}\n";
	print MAIL "Subject: $msub\n";
	print MAIL "MIME-Version: 1.0\n";
	print MAIL "Content-type: text/plain; charset=ISO-2022-JP\n";
	print MAIL "Content-Transfer-Encoding: 7bit\n";
	print MAIL "X-Mailer: $cf{version}\n\n";
	print MAIL "$body\n";
	close(MAIL);
}

#-----------------------------------------------------------
#  JPEGサイズ認識
#-----------------------------------------------------------
sub j_size {
	my $jpg = shift;
	
	my ($h, $w, $t);
	open(IMG,"$jpg");
	binmode(IMG);
	read(IMG, $t, 2);
	while (1) {
		read(IMG, $t, 4);
		my ($m, $c, $l) = unpack("a a n", $t);
		
		if ($m ne "\xFF") {
			$w = $h = 0;
			last;
		} elsif ((ord($c) >= 0xC0) && (ord($c) <= 0xC3)) {
			read(IMG, $t, 5);
			($h, $w) = unpack("xnn", $t);
			last;
		} else {
			read(IMG, $t, ($l - 2));
		}
	}
	close(IMG);
	
	return ($w, $h);
}

#-----------------------------------------------------------
#  GIFサイズ認識
#-----------------------------------------------------------
sub g_size {
	my $gif = shift;
	
	my $data;
	open(IMG,"$gif");
	binmode(IMG);
	sysread(IMG, $data, 10);
	close(IMG);
	
	if ($data =~ /^GIF/) { $data = substr($data, -4); }
	my $w = unpack("v", substr($data, 0, 2));
	my $h = unpack("v", substr($data, 2, 2));
	
	return ($w, $h);
}

#-----------------------------------------------------------
#  PNGサイズ認識
#-----------------------------------------------------------
sub p_size {
	my $png = shift;
	
	my $data;
	open(IMG,"$png");
	binmode(IMG);
	read(IMG, $data, 24);
	close(IMG);
	
	my $w = unpack("N", substr($data, 16, 20));
	my $h = unpack("N", substr($data, 20, 24));
	
	return ($w, $h);
}

#-----------------------------------------------------------
#  禁止ワードチェック
#-----------------------------------------------------------
sub no_wd {
	my $flg;
	foreach ( split(/,/,$cf{no_wd}) ) {
		if (index("$in{name} $in{sub} $in{comment}",$_) >= 0) {
			$flg = 1;
			last;
		}
	}
	if ($flg) { error("禁止ワードが含まれています"); }
}

#-----------------------------------------------------------
#  日本語チェック
#-----------------------------------------------------------
sub jp_wd {
	if ($in{comment} !~ /(?:[\xC0-\xDF][\x80-\xBF]|[\xE0-\xEF][\x80-\xBF]{2}|[\xF0-\xF7][\x80-\xBF]{3})/x) {
		error("メッセージに日本語が含まれていません");
	}
}

#-----------------------------------------------------------
#  URL個数チェック
#-----------------------------------------------------------
sub urlnum {
	my $com = $in{comment};
	my ($num) = ($com =~ s|(https?://)|$1|ig);
	if ($num > $cf{urlnum}) {
		error("コメント中のURLアドレスは最大$cf{urlnum}個までです");
	}
}

#-----------------------------------------------------------
#  アクセス制限
#-----------------------------------------------------------
sub get_host {
	# IP&ホスト取得
	my $host = $ENV{REMOTE_HOST};
	my $addr = $ENV{REMOTE_ADDR};
	
	if ($cf{gethostbyaddr} && ($host eq "" || $host eq $addr)) {
		$host = gethostbyaddr(pack("C4", split(/\./, $addr)), 2);
	}
	
	# IPチェック
	my $flg;
	foreach ( split(/\s+/, $cf{deny_addr}) ) {
		s/\./\\\./g;
		s/\*/\.\*/g;
		
		if ($addr =~ /^$_/i) { $flg = 1; last; }
	}
	if ($flg) {
		error("アクセスを許可されていません");
	
	# ホストチェック
	} elsif ($host) {
		
		foreach ( split(/\s+/, $cf{deny_host}) ) {
			s/\./\\\./g;
			s/\*/\.\*/g;
			
			if ($host =~ /$_$/i) { $flg = 1; last; }
		}
		if ($flg) {
			error("アクセスを許可されていません");
		}
	}
	if ($host eq "") { $host = $addr; }
	return ($host,$addr);
}

#-----------------------------------------------------------
#  crypt暗号
#-----------------------------------------------------------
sub encrypt {
	my $in = shift;
	
	my @wd = ('a' .. 'z', 'A' .. 'Z', 0 .. 9, '.', '/');
	my $salt = $wd[int(rand(@wd))] . $wd[int(rand(@wd))];
	crypt($in, $salt) || crypt ($in, '$1$' . $salt);
}

#-----------------------------------------------------------
#  crypt照合
#-----------------------------------------------------------
sub decrypt {
	my ($in,$dec) = @_;
	
	my $salt = $dec =~ /^\$1\$(.*)\$/ ? $1 : substr($dec, 0, 2);
	if (crypt($in, $salt) eq $dec || crypt($in, '$1$' . $salt) eq $dec) {
		return 1;
	} else {
		return 0;
	}
}

#-----------------------------------------------------------
#  完了メッセージ
#-----------------------------------------------------------
sub message {
	my ($msg) = @_;
	
	open(IN,"$cf{tmpldir}/mesg.html") or error("open err: mesg.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	$tmpl =~ s/!(bbs_cgi|bbs_title|cmnurl)!/$cf{$1}/g;
	$tmpl =~ s/!message!/$msg/g;
	
	print "Content-type: text/html; charset=utf-8\n\n";
	print $tmpl;
	exit;
}

#-----------------------------------------------------------
#  クッキー発行
#-----------------------------------------------------------
sub set_cookie {
	my @data = @_;
	
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,undef,undef) = gmtime(time + 60*24*60*60);
	my @mon  = qw|Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec|;
	my @week = qw|Sun Mon Tue Wed Thu Fri Sat|;
	
	# 時刻フォーマット
	my $gmt = sprintf("%s, %02d-%s-%04d %02d:%02d:%02d GMT",
				$week[$wday],$mday,$mon[$mon],$year+1900,$hour,$min,$sec);
	
	# URLエンコード
	my $cook;
	foreach (@data) {
		s/(\W)/sprintf("%%%02X", unpack("C", $1))/eg;
		$cook .= "$_<>";
	}
	
	print "Set-Cookie: $cf{cookie_id}=$cook; expires=$gmt\n";
}

#-----------------------------------------------------------
#  ファイルチェック
#-----------------------------------------------------------
sub check_upl {
	my ($ext) = @_;
	
	# mimeタイプ
	my $mime = $cgi->param_mime('upfile');
	
	# text
	my $flg;
	if ($ext eq '.txt' && $cf{ok_text} && $mime =~ /^text\/plain$/i) { $flg++; }
	elsif ($ext eq '.gif' && $cf{ok_gif} && $mime =~ /^image\/gif$/i) { check_gif(); $flg++; }
	elsif ($ext eq '.png' && $cf{ok_png} && $mime =~ /^image\/(x-)?png$/i) { check_png(); $flg++; }
	elsif ($ext eq '.jpg' && $cf{ok_jpeg} && $mime =~ /^image\/p?jpe?g$/i) { check_jpg(); $flg++; }
	elsif ($ext eq '.xls' && $cf{ok_excel} && $mime =~ /^application\/(vnd\.)?ms-?excel$/i) { $flg++; }
	elsif ($ext eq '.xlsx' && $cf{ok_excel} && $mime =~ /^application\/vnd\.openxmlformats-officedocument\.spreadsheetml\.sheet$/i) { $flg++; }
	elsif ($ext eq '.doc' && $cf{ok_word} && $mime =~ /^application\/(vnd\.)?ms-?word$/i) { $flg++; }
	elsif ($ext eq '.docx' && $cf{ok_word} && $mime =~ /^application\/vnd\.openxmlformats-officedocument\.wordprocessingml\.document$/i) { $flg++; }
	elsif ($ext eq '.ppt' && $cf{ok_ppt} && $mime =~ /^application\/(vnd\.)?ms-?powerpoint$/i) { $flg++; }
	elsif ($ext eq '.pptx' && $cf{ok_ppt} && $mime =~ /^application\/vnd\.openxmlformats-officedocument\.presentationml\.presentation$/i) { $flg++; }
	elsif ($ext eq '.zip' && $cf{ok_zip} && $mime =~ /^application\/(x-)?zip(-compressed)?$/i) { $flg++; }
	elsif ($ext eq '.pdf' && $cf{ok_pdf} && $mime =~ /^application\/pdf$/i) { $flg++; }
	
	if (!$flg) { error('このファイルは取り扱いできません'); }
}

#-----------------------------------------------------------
#  GIFチェック
#-----------------------------------------------------------
sub check_gif {
	# マジックバイト確認
	if ($in{upfile} !~ /^gif8[79]a/i) { error('このGIF画像は取扱できません!!'); }
}

#-----------------------------------------------------------
#  PNGチェック
#-----------------------------------------------------------
sub check_png {
	# XSS文字
	if ($in{upfile} =~ /script|embed|object|iframe|onclick|onblur|onchange|onmouse|onError|onload|onfocus|onselect|onsubmit|onunload|onreset|onabort|ondblclick|onkey|ondragdrop/i) {
		error('このPNG画像は取扱できません!');
	}
	# マジックバイト確認
	if ($in{upfile} !~ /^\x89PNG/i) { error('このPNG画像は取扱できません!!'); }
}

#-----------------------------------------------------------
#  JPEGチェック
#-----------------------------------------------------------
sub check_jpg {
	# マジックバイト確認
	if ($in{upfile} !~ /\xff\xd8/) { error('このJPEG画像は取扱できません!!'); }
}

#-----------------------------------------------------------
#  mimeエンコード
#  [quote] http://www.din.or.jp/~ohzaki/perl.htm#JP_Base64
#-----------------------------------------------------------
sub mime_unstructured_header {
  my $oldheader = shift;
  jcode::convert(\$oldheader,'euc','utf8');
  my ($header,@words,@wordstmp,$i);
  my $crlf = $oldheader =~ /\n$/;
  $oldheader =~ s/\s+$//;
  @wordstmp = split /\s+/, $oldheader;
  for ($i = 0; $i < $#wordstmp; $i++) {
    if ($wordstmp[$i] !~ /^[\x21-\x7E]+$/ and
	$wordstmp[$i + 1] !~ /^[\x21-\x7E]+$/) {
      $wordstmp[$i + 1] = "$wordstmp[$i] $wordstmp[$i + 1]";
    } else {
      push(@words, $wordstmp[$i]);
    }
  }
  push(@words, $wordstmp[-1]);
  foreach my $word (@words) {
    if ($word =~ /^[\x21-\x7E]+$/) {
      $header =~ /(?:.*\n)*(.*)/;
      if (length($1) + length($word) > 76) {
	$header .= "\n $word";
      } else {
	$header .= $word;
      }
    } else {
      $header = add_encoded_word($word, $header);
    }
    $header =~ /(?:.*\n)*(.*)/;
    if (length($1) == 76) {
      $header .= "\n ";
    } else {
      $header .= ' ';
    }
  }
  $header =~ s/\n? $//mg;
  $crlf ? "$header\n" : $header;
}
sub add_encoded_word {
  my ($str, $line) = @_;
  my $result;
  my $ascii = '[\x00-\x7F]';
  my $twoBytes = '[\x8E\xA1-\xFE][\xA1-\xFE]';
  my $threeBytes = '\x8F[\xA1-\xFE][\xA1-\xFE]';
  while (length($str)) {
    my $target = $str;
    $str = '';
    if (length($line) + 22 +
	($target =~ /^(?:$twoBytes|$threeBytes)/o) * 8 > 76) {
      $line =~ s/[ \t\n\r]*$/\n/;
      $result .= $line;
      $line = ' ';
    }
    while (1) {
      my $encoded = '=?ISO-2022-JP?B?' .
      b64encode(jcode::jis($target,'euc','z')) . '?=';
      if (length($encoded) + length($line) > 76) {
	$target =~ s/($threeBytes|$twoBytes|$ascii)$//o;
	$str = $1 . $str;
      } else {
	$line .= $encoded;
	last;
      }
    }
  }
  $result . $line;
}
# [quote] http://www.tohoho-web.com/perl/encode.htm
sub b64encode {
    my $buf = shift;
    my ($mode,$tmp,$ret);
    my $b64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                . "abcdefghijklmnopqrstuvwxyz"
                . "0123456789+/";
	
    $mode = length($buf) % 3;
    if ($mode == 1) { $buf .= "\0\0"; }
    if ($mode == 2) { $buf .= "\0"; }
    $buf =~ s/(...)/{
        $tmp = unpack("B*", $1);
        $tmp =~ s|(......)|substr($b64, ord(pack("B*", "00$1")), 1)|eg;
        $ret .= $tmp;
    }/eg;
    if ($mode == 1) { $ret =~ s/..$/==/; }
    if ($mode == 2) { $ret =~ s/.$/=/; }
    
    return $ret;
}

